<?php

namespace CLF\Inc\Common;

use CLF\Inc\Common\Interfaces\PaymentMethodInterface;
use WC_Countries;
use WC_Payment_Gateway;

abstract class BasePaymentMethod extends WC_Payment_Gateway implements PaymentMethodInterface
{
    const NOT_ALLOW_REFUND_ORDER_STATUSES = [
        'pending',
        'on-hold',
        'failed',
    ];
    	/**
	 * Enable for virtual products.
	 *
	 * @var bool
	 */
	public $enable_for_virtual;

    protected $gateway_code;
    public function __construct()
    {
        $this->setup_properties();
        $this->init_form_fields();
        $this->init_settings();

        $this->enabled              = $this->get_option( 'enabled', 'no' );
        $this->title                = $this->get_option( 'title', $this->get_method_title() );
        $this->description          = $this->get_option( 'description' );
        $this->countries            = $this->get_option( 'countries' );
        $this->errors               = array();
       
        $this->enable_for_virtual = $this->get_option( 'enable_for_virtual', 'yes' ) === 'yes';

        add_action(
            'woocommerce_update_options_payment_gateways_' . $this->id,
            array( $this, 'process_admin_options' )
        );
        add_action(
            'woocommerce_update_options_payment_gateways_' . $this->id,
            array( $this, 'display_errors' )
        );
    }
    protected function setup_properties() { 
        $this->id = $this->get_payment_method_id();
        $this->supports = array(
            'products',
            'refunds',
        );
        $this->method_title        = $this->get_payment_method_title() . ' (Clever Finance)';
        $this->method_description  = $this->get_payment_method_description();
        $this->gateway_code        = $this->get_payment_method_code();
        $this->icon                = $this->get_logo();
        $this->form_fields         = $this->add_form_fields();

    }
    public function getSettings() {
        return [
            'title' => $this->get_payment_method_title(),
            'description' => $this->get_payment_method_description(),
            'supports' => $this->supports,
            'logo' => $this->get_logo()
        ];
    }
    private function get_logo()
    {
        $icon = $this->get_payment_method_icon();

        return esc_url(plugins_url( '', __FILE__ ) . '/assets/public/images/' . $icon);
    }

    private function get_countries()
    {
        $countries = new WC_Countries();
        return $countries->get_allowed_countries();
    }

    public function get_payment_method_description() {
        return sprintf(
            /* translators: %1$: The payment method title */
            __( 'Pay with %1$s.', 'clever-finance-payments' ),
            $this->get_payment_method_title()
        );
    }

    public function add_form_fields() {
        return array(
            'enabled' => array(
                'title'   => __( 'Enable/Disable', 'clever-finance-payments' ),
                'type'    => 'checkbox',
                'label'   => sprintf(
                    /* translators: %1$: The payment method title */
                    __( 'Enable %1$s.', 'clever-finance-payments' ),
                    $this->get_payment_method_title()
                ),
                'default' => 'no',
            ),
            'title' => array(
                'title'       => __( 'Title', 'clever-finance-payments' ),
                'type'        => 'text',
                'description' => __( 'This controls the title which the user sees during checkout.', 'clever-finance-payments' ),
                'default'     => $this->get_payment_method_title(),
                'desc_tip'    => true,
            ),
            'description' => array(
                'title'       => __( 'Description', 'clever-finance-payments' ),
                'type'        => 'textarea',
                'description' => __( 'This controls the description which the user sees during checkout.', 'clever-finance-payments' ),
                'default'     => sprintf(
                    /* translators: %1$: The payment method title */
                    __( 'Pay with %1$s.', 'clever-finance-payments' ),
                    $this->get_payment_method_title()
                ),
                'desc_tip'    => true,
            ),
            'countries' => array(
                'title'       => __( 'Countries', 'clever-finance-payments' ),
                'type'        => 'multiselect',
                'class'       => 'wc-enhanced-select',
                'description' => __( 'Select the countries where this payment method should be available.', 'clever-finance-payments' ),
                'default'     => $this->get_option( 'countries', array() ),
                'desc_tip'    => true,
                'options'     => $this->get_countries(),
            ),
			'enable_for_virtual' => array(
				'title'   => __( 'Accept for virtual orders', 'woocommerce' ),
				'label'   => __( 'Accept if the order is virtual', 'woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'yes',
			),
        );
    }
   
    public function process_payment( $order_id ) {

        // Get the settings
        $settings = get_option( 'clever_finance_options' );
    
        // Get the order
        $order = wc_get_order( $order_id );
    
        // Get the remittance information
        $remittance_information = urlencode('Order #' . $order->get_id());
    
        // Get the payment internal ID
        $payment_internal_id = $order->get_order_key();
    
        // Get the order amount in cents
        $order_amount = $order->get_total() * 100;
    
        $payment_method = $this->get_payment_method_code();
    
        // Mark as pending payment
        $order->update_status( 'pending', __( 'Awaiting Digiteal payment', 'clever-finance-payments' ) );
    
        // Get the error page (checkout page with get parameter)
        $error_page = urlencode(wc_get_page_permalink( 'checkout') . '?digiteal_error=1');
    
        // Get the cancel page
        $cancel_page = urlencode(wc_get_page_permalink( 'checkout') . '?digiteal_cancel=1');
    
        // Get the success page
        $success_page = urlencode($this->get_return_url( $order ));
        error_log(print_r($success_page,1));
        // Define the identifier key
        $identifier_key = 'requesterVAT';
    
        // Check if the Digiteal VAT contains a colon
        if (isset($settings['clever_finance_vat']) && strpos($settings['clever_finance_vat'], ':') !== false) {
            // Set the identifier key to requesterIdentificationNumber
            $identifier_key = 'requesterIdentificationNumber';
        }
    
        // Validate required credentials
        if (empty($settings['clever_finance_login']) || empty($settings['clever_finance_password'])) {
            wc_get_logger()->error('Clever Finance API credentials missing.');
            error_log("Clever Finance API credentials missing.");
            error_log(print_r($settings, true));            
            return [
                'result'   => 'failure',
            ];
        }
    
        $long_payment_url = $this->get_api_url( $settings ) . "/payment-request/pay-button/execute?{$identifier_key}={$settings['clever_finance_vat']}&errorURL={$error_page}&cancelURL={$cancel_page}&confirmationURL={$success_page}&multiple=true&remittanceInfo={$remittance_information}&paymentInternalId={$payment_internal_id}&amountInCents={$order_amount}&paymentMethod={$payment_method}";
    
        // Check if the IBAN in the settings is valid
        if (!empty($settings['clever_finance_iban'])) {
            $long_payment_url .= "&iban={$settings['clever_finance_iban']}";
        }
    
        // Define the request options
        $options = [
            'body' => json_encode(['longURL' => $long_payment_url]),
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Basic ' . base64_encode($settings['clever_finance_login'] . ':' . $settings['clever_finance_password']),
            ],
        ];
    
        // Log the request URL
        wc_get_logger()->debug('Sending request to API: ' . $this->get_api_url( $settings ) . '/shortLink');
        wc_get_logger()->debug('Request Body: ' . json_encode($options));
    
        // Send the request
        $response = wp_remote_post($this->get_api_url( $settings ) . '/shortLink', $options);
       

        // Check if the request failed
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            wc_get_logger()->error('Payment API request failed: ' . $error_message);
            $this->errors[] = $error_message;
            return [
                'result'   => 'failure',
            ];
        }
    
        // Check if the response is not a 200
        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code !== 200) {
            $error_message = wp_remote_retrieve_response_message($response);
            wc_get_logger()->error('Payment API returned unexpected response code: ' . $response_code . ' - ' . $error_message);
            $this->errors[] = $error_message;
            return [
                'result'   => 'failure',
            ];
        }
    
        // Get the response body
        $response_body = wp_remote_retrieve_body($response);
    
        // Log the raw API response
        wc_get_logger()->debug('Raw API Response: ' . $response_body);
    
        // Decode JSON response
        $decoded_response = json_decode($response_body, true);
        // Check if shortURL exists
        if (!isset($decoded_response['shortURL'])) {
            wc_get_logger()->error('Payment API response missing shortURL. Full response: ' . print_r($decoded_response, true));
            return [
                'result'   => 'failure',
            ];
        }
        return [
            'result'   => 'success',
            'redirect' => $decoded_response['shortURL'],
        ];
    }
    

	private function get_api_url( $settings ) {
		$api_suffix = '/api/v1';

		if ( isset( $settings['clever_finance_test_mode'] ) && $settings['clever_finance_test_mode'] == 1 ) {
			return 'https://test.digiteal.eu' . $api_suffix;
		}

		return 'https://app.digiteal.eu' . $api_suffix;
	}
}
