<?php

namespace CLF\Inc\Admin;

class Clever_Finance_Admin
{
    private $connection_status = '';
    private $cached_valid_response = null;

    public function __construct() {}

    public function add_admin_menu()
    {
        if (!current_user_can('manage_options')) {
            return;
        }
    
        add_submenu_page(
            'woocommerce',
            'Clever Finance Settings',
            'Clever Finance',
            'manage_options',
            'clever-finance-settings',
            [$this, 'settings_page']
        );
    }

    public function register_settings()
    {
        register_setting(
            'clever_finance_options',
            'clever_finance_options',
            array($this, 'validate_and_check_settings')
        );

        add_settings_section(
            'clever_finance_settings_section',
            'Clever Finance Settings',
            null,
            'clever_finance_settings'
        );

        $settings = [
            'clever_finance_login' => 'Login',
            'clever_finance_password' => 'Password',
            'clever_finance_test_mode' => 'Enable Test Mode',
            'clever_finance_vat' => 'VAT',
            'clever_finance_iban' => 'IBAN',
        ];

        foreach ($settings as $key => $label) {
            add_settings_field(
                $key,
                $label,
                array($this, 'render_settings_field'),
                'clever_finance_settings',
                'clever_finance_settings_section',
                ['label_for' => $key, 'key' => $key]
            );
        }
    }

    public function render_settings_field($args)
    {
        $options = get_option('clever_finance_options');
        $key = $args['key'];
        $value = isset($options[$key]) ? esc_attr($options[$key]) : '';
        
        if ($key === 'clever_finance_test_mode') {
            echo '<input type="checkbox" id="' . esc_attr($key) . '" name="clever_finance_options[' . esc_attr($key) . ']" value="1" ' . checked(1, $value, false) . ' />';
        } else {
            $disabled = (in_array($key, ['clever_finance_login', 'clever_finance_password']) && $this->getConnection() ) ? 'disabled' : '';
            $type = ($key === 'clever_finance_password') ? 'password' : 'text';
            echo '<input type="' . $type . '" id="' . esc_attr($key) . '" name="clever_finance_options[' . esc_attr($key) . ']" value="' . $value . '" class="form-input"'.$disabled.' />';
        }
    }

    private function check_settings_actions() {
		if ( ! isset( $_REQUEST[ 'action' ] ) || $_REQUEST[ 'action' ] !== 'logout' ) {
			return;
		}

		// Get the settings
		$settings = get_option( 'clever_finance_options' );

		// Maybe deregister the webhook
		$webhook_deregistered = $this->maybe_deregister_webhook( $settings );

		// Check if the webhook was not deregistered
		if ( ! $webhook_deregistered ) {
			// Set the error message
			add_settings_error(
				'clever_finance_options',
				'clever_finance_options' . '_webhook_deregister_error',
				__( 'An error occurred while deregistering the webhook. Please try again.', 'clever-finance-settings' ),
				'error'
			);

			// Redirect to the settings page
			wp_redirect( admin_url( 'admin.php?page=clever-finance-settings' ) );

			// Return early
			return;
		}

		// Remove the credentials
		unset( $settings['clever_finance_login'] );
		unset( $settings['clever_finance_password'] );

		// Remove the VAT number
		unset( $settings['clever_finance_vat'] );

		// Remove the available payment methods
		unset( $settings['available_payment_methods'] );

		// Update the settings
		update_option( 'clever_finance_options', $settings );

		// Redirect to the settings page
		wp_redirect( admin_url( 'admin.php?page=clever-finance-settings' ) );
	}

    public function settings_page()
    {
        $this->check_settings_actions();
        $this->connection_status = $this->getConnection() ? 'Connected' : 'Not Connected';
        require_once plugin_dir_path(dirname(__FILE__)) . 'admin/views/settings-page.php';
    }

    private function getConnection(): bool
    {
        $settings = get_option('clever_finance_options', []);
        return !empty($settings['clever_finance_webhook_registered']);
    }
    public function validate_and_check_settings($input)
    {
        $validated = [];
        if (!empty($input['clever_finance_login'])) {
            $validated['clever_finance_login'] = sanitize_text_field($input['clever_finance_login']);
        }

        if (!empty($input['clever_finance_password'])) {
            $validated['clever_finance_password'] = sanitize_text_field($input['clever_finance_password']);
        }

        $validated['clever_finance_test_mode'] = isset($input['clever_finance_test_mode']) ? 1 : 0;

        if (!empty($input['clever_finance_vat'])) {
            $validated['clever_finance_vat'] = sanitize_text_field($input['clever_finance_vat']);
        }

        $this->check_settings($validated);
        return $validated;
    }

    public function check_settings(&$settings)
    {
        $vat_number = $this->validate_credentials($settings, true);

        if (!$vat_number) {
            add_settings_error(
                'clever_finance_options',
                'VAT',
                __('Invalid credentials.', 'clever_finance_payments')
            );
            $settings['clever_finance_login'] = '';
            $settings['clever_finance_password'] = '';
            return;
        }

        if (empty($settings['clever_finance_vat'])) {
            $settings['clever_finance_vat'] = $vat_number;
        }

        $this->get_available_payment_methods($settings);

        if (!$this->maybe_register_webhook($settings)) {
            add_settings_error(
                'clever_finance_options',
                'WebHook',
                __('Your domain seems local. Digiteal checks if the domain can be reached remotely.', 'clever_finance_payments')
            );
        }
    }

    private function get_api_url(): string
    {
        $settings = get_option('clever_finance_options', []);
        $api_base = isset($settings['clever_finance_test_mode']) && $settings['clever_finance_test_mode'] == 1
            ? 'https://test.digiteal.eu'
            : 'https://app.digiteal.eu';

        return $api_base . '/api/v1';
    }

    private function get_auth_headers($settings): array
    {
        return [
            'Content-Type' => 'application/json',
            'Authorization' => 'Basic ' . base64_encode($settings['clever_finance_login'] . ':' . $settings['clever_finance_password']),
        ];
    }

    private function validate_credentials($settings, bool $return_vat_number = false)
    {
        if ($this->cached_valid_response !== null) {
            return $this->cached_valid_response;
        }

        $response = wp_remote_get($this->get_api_url() . '/webhook', ['headers' => $this->get_auth_headers($settings)]);
        $response_code = wp_remote_retrieve_response_code($response);

        if ($response_code !== 200) {
            $this->cached_valid_response = false;
            return false;
        }

        $response_body = json_decode(wp_remote_retrieve_body($response));

        if (isset($response_body->errorCode)) {
            $this->cached_valid_response = false;
            return false;
        }

        $this->cached_valid_response = true;

        return $return_vat_number ? ($response_body->vatNumber ?? $response_body->identificationNumber) : true;
    }

    private function get_available_payment_methods(&$settings)
    {
        $parameter_name = (strpos($settings['clever_finance_vat'], ':') !== false) ? 'identificationNumber' : 'vatNumber';
        $url = $this->get_api_url() . "/integrator/company-info?$parameter_name=" . $settings['clever_finance_vat'];

        $response = wp_remote_get($url, ['headers' => $this->get_auth_headers($settings)]);
        if (wp_remote_retrieve_response_code($response) !== 200) return;

        $response_body = json_decode(wp_remote_retrieve_body($response));

        $settings['available_payment_methods'] = $response_body->paymentMethods ?? [];
    }

    public function maybe_deregister_webhook(&$settings): bool
    {
        if (empty($settings['clever_finance_webhook_registered'])) {
            return true;
        }

        $url = $this->get_api_url() . '/webhook/PAYMENT_INITIATED';
        $response = wp_remote_request($url, ['headers' => $this->get_auth_headers($settings), 'method' => 'DELETE']);
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        if (!in_array($response_code, [200, 403], true) && strpos($response_body, 'WEBHOOK_NOT_FOUND') === false) {
            return false;
        }

        $settings['clever_finance_webhook_registered'] = false;
        return true;
    }

    private function maybe_register_webhook(&$settings): bool
    {
        if (!empty($settings['clever_finance_webhook_registered'])) {
            return true;
        }

        $webhook_url = get_rest_url(null, '/digiteal/v1/payment');
        $body = json_encode(['webHooks' => [['type' => 'PAYMENT_INITIATED', 'url' => $webhook_url]]]);

        $response = wp_remote_post($this->get_api_url() . '/webhook/configuration', [
            'body' => $body,
            'headers' => $this->get_auth_headers($settings),
        ]);
        if (wp_remote_retrieve_response_code($response) !== 200) {
            return false;
        }

        $settings['clever_finance_webhook_registered'] = true;
        return true;
    }
}
