<?php

/**
 * 2007-2025 PrestaShop
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to http://www.prestashop.com for more information.
 *
 *  @author    PrestaShop SA <contact@prestashop.com>
 *  @copyright 2007-2025 PrestaShop SA
 *  @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
 *  International Registered Trademark & Property of PrestaShop SA
 */

if (!defined('_PS_VERSION_')) {
    exit;
}
require_once _PS_MODULE_DIR_ . '/clfpayment/src/includeClasses.php';

class ClfPayment extends PaymentModule
{

    protected $config_form = false;
    public $limited_currencies = array();
    public $is_eu_compatible;
    private $currentController;
    private $companyStatus;

    const MODULE_MIN_VERSION = '1.5.0';
    const MODULE_MAX_VERSION = '9.0.0';

    const REINITIALIZE_MODULE = 'fds0frk34kv';
    const SUBMIT_CREDENTIAL = 'g1fds56g4s3dh1';


    public function __construct()
    {
        $this->name = 'clfpayment';
        $this->tab = 'payments_gateways';
        $this->version = '1.0.0';
        $this->author = 'Clever Finance';
        $this->need_instance = 1;
        $this->controllers = ['redirect', 'confirmation', 'error'];


        /**
         * Set $this->bootstrap to true if your module is compliant with bootstrap (PrestaShop 1.6)
         */
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('Clever Finance Payments');
        $this->description = $this->l('Clever Finance payment module for PrestaShop.');
        $this->currentController = Tools::strtolower(Tools::getValue('controller'));

        $this->confirmUninstall = $this->l('');

        $this->is_eu_compatible = 1;



        $this->limited_countries = ['USA', 'UK'];
        $this->ps_versions_compliancy = ['min' => self::MODULE_MIN_VERSION, 'max' => self::MODULE_MAX_VERSION];
        parent::__construct();
        try { // Mainly used for P1.5
            $id_order = Tools::getValue('id_order');
            if (false !== $id_order) {
                $order = new Order((int) $id_order);
                if (($order->module == $this->name) && ($this->context->controller instanceof OrderConfirmationController)) {
                    $this->displayName = $order->payment;
                }
            }
        } catch (Exception $e) {
        }
    }

    /**
     * Don't forget to create update methods if needed:
     * http://doc.prestashop.com/display/PS16/Enabling+the+Auto-Update
     */
    public function install()
    {
        if (version_compare(_PS_VERSION_, self::MODULE_MIN_VERSION, '<')) {
            return false;
        }
        if (extension_loaded('curl') == false) {
            $this->_errors[] = $this->l('You have to enable the cURL extension on your server to install this module');
            return false;
        }

        Configuration::updateValue('CLFPAYMENT_LIVE_MODE', false);
        if (version_compare(_PS_VERSION_, 1.7, '<')) {
            $installed = parent::install() &&
                $this->registerHook('header') &&
                $this->registerHook('backOfficeHeader') &&
                $this->registerHook('paymentReturn');
        } else {
            $installed = parent::install() &&
                $this->registerHook('displayHeader') &&
                $this->registerHook('displayBackOfficeHeader') &&
                $this->registerHook('paymentReturn');
        }

        if ($installed) {
            $installed = Configuration::updateValue('CLF_ENABLE_LOGGER', 0) &&
                Configuration::updateValue('CLF_MODE', 0) &&
                Configuration::updateValue('CLF_SPLIT_PAYMENT_METHODE', true);
        }
        if ($installed) {
            (false !== ($dc = CLFRest::getConf())) ?
                $installed = Configuration::updateValue('CLF_KPIID', $dc) : $installed = false;
        }
        if ($installed) {
            (false !== ($dc = CLFRest::getConf(true))) ?
                $installed = Configuration::updateValue('CLF_KPIIDT', $dc) : $installed = false;
        }

        if ($installed) {
            if (version_compare(_PS_VERSION_, '1.7', '<')) {
                if (!$this->registerHook('payment') || !$this->registerHook('displayPaymentEU')) {
                    CLFLogger::logError('Hook payment or displayPaymentEU could not be saved.');
                    $this->_errors[] = $this->l('One or more hooks required for the module could not be saved.');
                    $installed = false;
                }
            } else {
                if (!$this->registerHook('paymentOptions')) {
                    CLFLogger::logError('Hook paymentOptions could not be saved.');
                    $this->_errors[] = $this->l('One or more hooks required for the module could not be saved.');
                    $installed = false;
                }
            }
        }

        if (false === $installed) {
            $this->uninstall();

            return false;
        }

        return (bool) $installed;
    }

    public function uninstall()
    {
        Configuration::deleteByName('CLFPAYMENT_LIVE_MODE');
        if (version_compare(_PS_VERSION_, '1.7', '<')) {
            $uninstalled = parent::uninstall() &&
                $this->unregisterHook('header') &&
                $this->unregisterHook('backOfficeHeader') &&
                $this->unregisterHook('paymentReturn');
        } else {
            $uninstalled = parent::uninstall() &&
                $this->unregisterHook('displayHeader') &&
                $this->unregisterHook('displayBackOfficeHeader') &&
                $this->unregisterHook('paymentReturn');
        }

        if ($uninstalled) {
            $uninstalled &= Configuration::deleteByName('CLF_ENABLE_LOGGER') &&
                Configuration::deleteByName('CLF_ROADMAP') &&
                Configuration::deleteByName('CLF_ENABLE_ROADMAP') &&
                Configuration::deleteByName('CLF_KPIID') &&
                Configuration::deleteByName('CLF_KPIIDT') &&
                Configuration::deleteByName('CLF_MODE') &&
                Configuration::deleteByName('CLF_SPLIT_PAYMENT_METHODE');
            $uninstalled &= Db::getInstance()->execute(
                'DELETE FROM `' . _DB_PREFIX_ . "configuration` WHERE `name` LIKE 'CLF_%'"
            );
        }

        if ($uninstalled) {
            $uninstalled = $this->uninstallTab();
        }

        return (bool) $uninstalled;
    }
    private function uninstallTab()
    {
        $idTab = (int) Tab::getIdFromClassName('AdminCLFRoadmap');
        if ($idTab) {
            $tab = new Tab($idTab);
            return $tab->delete();
        }

        return true;
    }
    /**
     * Load the configuration form
     */
    public function getContent()
    {
        $this->companyStatus = new CLFCompanyStatus;
        $smartyVars = [
            'CLF_description'      =>  $this->description,
            'form_action'          => AdminController::$currentIndex . '&configure=' . $this->name . '&token=' . Tools::getAdminTokenLite('AdminModules'),
            'reinit_module'        => AdminController::$currentIndex . '&configure=' . $this->name . '&' . self::REINITIALIZE_MODULE . '&token=' . Tools::getAdminTokenLite('AdminModules'),
            'reinit_submit'        => self::REINITIALIZE_MODULE,
        ];
        if (Tools::isSubmit(self::REINITIALIZE_MODULE)) {
            $this->companyStatus->emptyAndReload();
            $smartyVars['messageSuccess'] = $this->l('The module has been successfully reset.');
        } elseif (Tools::isSubmit(self::SUBMIT_CREDENTIAL)) {
            if (Tools::getIsset('clfemail') && Tools::getIsset('clfpassword')) {
                $clfmode = (int) (Tools::getValue('clfmode') == 0 ? 0 : 1);
                $this->companyStatus->setMode($clfmode);
                $this->companyStatus->setEmail(Tools::getValue('clfemail'));
                $this->companyStatus->setPassword(Tools::getValue('clfpassword'));
                $this->companyStatus->setVatNumber(Tools::getValue('clfvatNumber'));
                $this->companyStatus->setIBANNumber(Tools::getValue('clfibanNumber'));
                $this->companyStatus->save();
            }

            if ($this->companyStatus->getModuleReady()) {
            } else {
                $shop_url = Tools::getCurrentUrlProtocolPrefix() . htmlspecialchars($_SERVER['HTTP_HOST'], ENT_COMPAT, 'UTF-8') . __PS_BASE_URI__;
                $webhookValidationLink = $shop_url . 'modules/digiteal/validation.php';
                $webhookErrorLink = $shop_url . 'modules/digiteal/error.php';
                if ($this->companyStatus->generateWebhookConfiguration(
                    $webhookValidationLink,
                    $webhookErrorLink,
                    Tools::getValue('clfemail'),
                    Tools::getValue('clfpassword')
                )) {
                    $postPaymentMethods = [];
                    $paymentMethods = CLFPaymentMethod::getMethods();
                    foreach ($paymentMethods as $paymentMethod) {
                        if (Tools::getIsset('paymentMethods_' . strtolower($paymentMethod))) {
                            $postPaymentMethods[] = strtoupper(Tools::getValue('paymentMethods_' . strtolower($paymentMethod)));
                        }
                    }
                    $postPaymentMethods = json_encode($postPaymentMethods);
                    $this->companyStatus->setPaymentMethods($postPaymentMethods);
                    $this->companyStatus->setWebhookValidationUrl($webhookValidationLink);
                    $this->companyStatus->setWebhookErrorUrl($webhookErrorLink);
                    $this->companyStatus->setModuleReady(true);
                    $this->companyStatus->save();
                } else {
                    $smartyVars['messageError'] = $this->l('Please, make sure you have filled in the right information.');
                }
                if ($this->companyStatus->checkRestStatus()) {
                    $smartyVars['messageSuccess'] = $this->l('The information has been updated.');
                    if (Tools::getIsset('selectedIban')) {
                        $ibans = $this->companyStatus->getIbansAsArray();
                        $iban = Tools::getValue('selectedIban');
                        if (in_array($iban, $ibans)) {
                            $this->companyStatus->setSelectedIban($iban);
                            $this->companyStatus->save();
                        }
                    }
                    $splitPaymentMethods = Tools::getValue('splitPaymentMethods', true);
                    Configuration::updateValue('CLF_SPLIT_PAYMENT_METHODE', (bool) $splitPaymentMethods);
                } else {
                    $smartyVars['messageError'] = $this->l('Check Company Status failed.');
                }
            }
        }
        $smartyVars['clfmode'] = $this->companyStatus->getMode();
        $smartVars['connected'] = $this->companyStatus->isConnected();
        $smartyVars['submit_name'] = self::SUBMIT_CREDENTIAL;
        $smartyVars['submit_label'] = $this->l('Start the status check');
        $smartyVars['inputs'] = $this->getInputs();


        $this->smarty->assign($smartyVars);

        return $this->display(__FILE__, 'views/templates/admin/settings.tpl');
    }

    /**
     * @return array[]
     */
    protected function getInputs()
    {
        return [
            [
                'type'    => ((version_compare(_PS_VERSION_, '1.6', '<')) ? 'radio' : 'switch'),
                'label'   => $this->l('Mode Test'),
                'name'    => 'clfmode',
                'is_bool' => true,
                'value'   => $this->companyStatus->getMode(),
                'values'  => [
                    [
                        'id'    => 'clfmode_on',
                        'value' => 1,
                        'label' => $this->l('Enable'),
                    ],
                    [
                        'id'    => 'clfmode_off',
                        'value' => 0,
                        'label' => $this->l('Disable'),
                    ],
                ],
            ],
            [
                'type'     => 'text',
                'label'    => $this->l('Email'),
                'name'     => 'clfemail',
                'desc'     => $this->l(''),
                'value'    => $this->companyStatus->getEmail(),
                'required' => true,
            ],
            [
                'type'     => 'password',
                'label'    => $this->l('Password'),
                'name'     => 'clfpassword',
                'desc'     => $this->l(''),
                'value'    => $this->companyStatus->getPassword(),
                'required' => true,
            ],
            [
                'type'     => 'text',
                'label'    => $this->l('VAT number'),
                'name'     => 'clfvatNumber',
                'desc'     => $this->l('The intra-community VAT number of your company.'),
                'value'    => $this->companyStatus->getVatNumber(),
                'required' => false,
            ],

        ];
    }



    public function hookHeader($params)
    {
        $controller = $this->context->controller;
        if ($this->currentController === 'order' || $this->currentController === 'order-opc') {
            // If there is errors during payment and the customer is redirected to the checkout,
            // then keep the cart and display the message to the customer.
            if (isset($this->context->cookie->CLFErrors)) {
                $controller->errors = array_merge(
                    $controller->errors,
                    explode("\n", $this->context->cookie->CLFErrors)
                );
                unset($this->context->cookie->CLFErrors);
            }
        }
        $this->context->controller->addJS($this->_path . 'views/js/front.js');
    }

    /**
     * @param array $params
     */
    public function hookDisplayHeader($params)
    {
        $this->hookHeader($params);
    }

    /**
     * @param array $params
     */
    public function hookBackOfficeHeader($params)
    {
        $moduleName = Tools::getValue('module_name');
        $moduleName = (false === $moduleName ? Tools::getValue('configure') : $moduleName);

        if ($moduleName === $this->name) {
            $this->context->controller->addJS($this->_path . 'views/js/back.js');
            $this->context->controller->addCSS($this->_path . 'views/css/back.css');
            if (version_compare(_PS_VERSION_, '1.6', '<')) {
                $this->context->controller->addCSS($this->_path . 'views/css/back-1.5.css');
            }
        }
    }

    public function hookDisplayBackOfficeHeader($params)
    {
        $this->hookBackOfficeHeader($params);
    }

    /**
     * @param array $params
     */
    public function hookPaymentOptions($params)
    {
        if (!$this->active) {
            return;
        }
        if (!$this->checkCurrency($params['cart'])) {
            return;
        }

        $this->companyStatus = new CLFCompanyStatus();
        if (!$this->companyStatus->getModuleReady() || !$this->companyStatus->canReceiveFunds() || !$this->companyStatus->hasSelectedIban() || count($this->companyStatus->getPaymentMethodsAsArray()) <= 0) {
            return;
        }

        if (Configuration::get('CLF_SPLIT_PAYMENT_METHODE')) {
            return $this->getPaymentOptions();
        } else {
            $CLFPaymentOption = new PrestaShop\PrestaShop\Core\Payment\PaymentOption();
            $CLFPaymentOption->setModuleName($this->name)
                ->setCallToActionText($this->l('Pay online'))
                ->setAction($this->context->link->getModuleLink($this->name, 'redirect', [], true));

            $logo = $this->getPaymentLogo();
            if (!is_null($logo)) {
                $CLFPaymentOption->setLogo(Media::getMediaPath($logo));
            }

            return [$CLFPaymentOption];
        }
    }

    /**
     * @param array $params
     */
    public function hookPaymentReturn($params)
    {
        if (version_compare(_PS_VERSION_, '1.7', '<')) {
            $order = isset($params['order']) ? $params['order'] : $params['objOrder'];

            if (!$this->active || ($order->module != $this->name)) {
                return;
            }

            $state = $order->getCurrentState();
            if (in_array($state, [Configuration::get('PS_OS_PAYMENT'), Configuration::get('PS_OS_OUTOFSTOCK'), Configuration::get('PS_OS_OUTOFSTOCK_UNPAID')])) {
                $smartVars = [
                    'total_to_pay' => Tools::displayPrice($params['total_to_pay'], $params['currencyObj'], false),
                    'status'       => 'ok',
                    'id_order'     => $order->id,
                ];
                if (isset($order->reference) && !empty($order->reference)) {
                    $smartVars['reference'] = $order->reference;
                }
                $this->smarty->assign($smartVars);
            } else {
                $this->smarty->assign('status', 'failed');
            }

            return $this->display(__FILE__, 'payment_return.tpl');
        }
    }

    /**
     * @param $params
     *
     * @return array|void
     */
    public function hookDisplayPaymentEU($params)
    {
        if (!$this->active) {
            return;
        }

        // Currency support.
        if (!$this->checkCurrency($params['cart'])) {
            return;
        }

        $this->companyStatus = new CLFCompanyStatus();
        if (!$this->companyStatus->getModuleReady() || !$this->companyStatus->canReceiveFunds() || !$this->companyStatus->hasSelectedIban() || count($this->companyStatus->getPaymentMethodsAsArray()) <= 0) {
            return;
        }

        $payment_options = [
            'cta_text' => $this->l('Pay online'),
            'action'   => $this->context->link->getModuleLink($this->name, 'redirect', [], true),
        ];
        $logo = $this->getPaymentLogo();

        if (!is_null($logo)) {
            $payment_options['logo'] = $logo;
        }

        return $payment_options;
    }

    public function hookPayment($params)
    {
        if (!$this->active) {
            return;
        }

        // Currency support.
        if (!$this->checkCurrency($params['cart'])) {
            return;
        }

        $this->companyStatus = new CLFCompanyStatus();
        if (!$this->companyStatus->getModuleReady() || !$this->companyStatus->canReceiveFunds() || !$this->companyStatus->hasSelectedIban() || count($this->companyStatus->getPaymentMethodsAsArray()) <= 0) {
            return;
        }

        $payment_options = [
            'cta_text' => $this->l('Pay online'),
            'action'   => $this->context->link->getModuleLink($this->name, 'redirect', [], true),
        ];
        $logo = $this->getPaymentLogo();
        if (!is_null($logo)) {
            $payment_options['logo'] = $logo;
        }

        $this->smarty->assign($payment_options);

        return $this->display(__FILE__, 'payment.tpl');
    }
    /**
     * @param $cart
     *
     * @return bool
     */
    public function checkCurrency($cart)
    {
        $currency_order = new Currency($cart->id_currency);
        $currencies_module = $this->getCurrency($cart->id_currency);
        if (is_array($currencies_module)) {
            foreach ($currencies_module as $currency_module) {
                if ($currency_order->id == $currency_module['id_currency']) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Get the payment options available based on the company's status.
     *
     * @return array An array of PaymentOption objects representing the available payment methods
     */
    private function getPaymentOptions()
    {
        $paymentMethods = $this->companyStatus->getPaymentMethodsAsArray();
        if (count($paymentMethods) > 0) {
            $CLFPaymentOptions = [];
            foreach ($paymentMethods as $paymentMethod) {
                $CLFPaymentOption = new PrestaShop\PrestaShop\Core\Payment\PaymentOption();
                $CLFPaymentOption->setModuleName($this->name)
                    ->setCallToActionText($this->l('Pay by') . ' ' . strtolower(str_replace('_', ' ', $paymentMethod)))
                    ->setAction($this->context->link->getModuleLink($this->name, 'redirect', ['paymentMethod' => $paymentMethod], true));

                $logo = $this->getPaymentLogo($paymentMethod);
                if (!is_null($logo)) {
                    $CLFPaymentOption->setLogo(Media::getMediaPath($logo));
                }
                $CLFPaymentOptions[] = $CLFPaymentOption;
            }

            return $CLFPaymentOptions;
        }

        return [];
    }

    /**
     * @return null|string payment logo URL
     */
    protected function getPaymentLogo($paymentMethod = null)
    {
        if (isset($this->companyStatus)) {
            $payment_logos = $this->companyStatus->getPaymentMethodsLogos();
            if (!is_null($payment_logos)) {
                if (is_null($paymentMethod)) {
                    $file = _PS_MODULE_DIR_ . $this->name . '/views/img/logos/' . $payment_logos . '.svg';
                    if (!file_exists($file)) {
                        $payment_methods = $this->companyStatus->getPaymentMethodsAsArray();
                        if (count($payment_methods) > 0) {
                            CLFPaymentMethod::generateLogosFile($file, $payment_methods);
                        }
                    }
                    if (version_compare(_PS_VERSION_, '1.6', '<')) {
                        return Tools::getProtocol() . Tools::getMediaServer($file) . '/modules/' . $this->name . '/views/img/logos/' . $payment_logos . '.svg';
                    } else {
                        return Media::getMediaPath($file);
                    }
                } else {
                    $filename = CLFPaymentMethod::getPaymentMethodLogo($paymentMethod);
                    $file = _PS_MODULE_DIR_ . $this->name . '/views/img/logos/' . $filename;
                    if (version_compare(_PS_VERSION_, '1.6', '<')) {
                        return Tools::getProtocol() . Tools::getMediaServer($file) . '/modules/' . $this->name . '/views/img/logos/' . $filename;
                    } else {
                        return Media::getMediaPath($file);
                    }
                }
            }
        }

        return null;
    }
}
